/* unpack a hdf file
 *
 * 2003-02-12T06:57:49Z
 *   works
 *
 * 2003-02-06T23:11:25Z
 */
#include "stdafx.h"
#include "hdf.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

typedef unsigned int uint;
typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned long u32;

static u16 table_crc16[256] = {
	0x0000,0xc0c1,0xc181,0x0140,0xc301,0x03c0,0x0280,0xc241,0xc601,0x06c0,0x0780,0xc741,0x0500,0xc5c1,0xc481,0x0440,
	0xcc01,0x0cc0,0x0d80,0xcd41,0x0f00,0xcfc1,0xce81,0x0e40,0x0a00,0xcac1,0xcb81,0x0b40,0xc901,0x09c0,0x0880,0xc841,
	0xd801,0x18c0,0x1980,0xd941,0x1b00,0xdbc1,0xda81,0x1a40,0x1e00,0xdec1,0xdf81,0x1f40,0xdd01,0x1dc0,0x1c80,0xdc41,
	0x1400,0xd4c1,0xd581,0x1540,0xd701,0x17c0,0x1680,0xd641,0xd201,0x12c0,0x1380,0xd341,0x1100,0xd1c1,0xd081,0x1040,
	0xf001,0x30c0,0x3180,0xf141,0x3300,0xf3c1,0xf281,0x3240,0x3600,0xf6c1,0xf781,0x3740,0xf501,0x35c0,0x3480,0xf441,
	0x3c00,0xfcc1,0xfd81,0x3d40,0xff01,0x3fc0,0x3e80,0xfe41,0xfa01,0x3ac0,0x3b80,0xfb41,0x3900,0xf9c1,0xf881,0x3840,
	0x2800,0xe8c1,0xe981,0x2940,0xeb01,0x2bc0,0x2a80,0xea41,0xee01,0x2ec0,0x2f80,0xef41,0x2d00,0xedc1,0xec81,0x2c40,
	0xe401,0x24c0,0x2580,0xe541,0x2700,0xe7c1,0xe681,0x2640,0x2200,0xe2c1,0xe381,0x2340,0xe101,0x21c0,0x2080,0xe041,
	0xa001,0x60c0,0x6180,0xa141,0x6300,0xa3c1,0xa281,0x6240,0x6600,0xa6c1,0xa781,0x6740,0xa501,0x65c0,0x6480,0xa441,
	0x6c00,0xacc1,0xad81,0x6d40,0xaf01,0x6fc0,0x6e80,0xae41,0xaa01,0x6ac0,0x6b80,0xab41,0x6900,0xa9c1,0xa881,0x6840,
	0x7800,0xb8c1,0xb981,0x7940,0xbb01,0x7bc0,0x7a80,0xba41,0xbe01,0x7ec0,0x7f80,0xbf41,0x7d00,0xbdc1,0xbc81,0x7c40,
	0xb401,0x74c0,0x7580,0xb541,0x7700,0xb7c1,0xb681,0x7640,0x7200,0xb2c1,0xb381,0x7340,0xb101,0x71c0,0x7080,0xb041,
	0x5000,0x90c1,0x9181,0x5140,0x9301,0x53c0,0x5280,0x9241,0x9601,0x56c0,0x5780,0x9741,0x5500,0x95c1,0x9481,0x5440,
	0x9c01,0x5cc0,0x5d80,0x9d41,0x5f00,0x9fc1,0x9e81,0x5e40,0x5a00,0x9ac1,0x9b81,0x5b40,0x9901,0x59c0,0x5880,0x9841,
	0x8801,0x48c0,0x4980,0x8941,0x4b00,0x8bc1,0x8a81,0x4a40,0x4e00,0x8ec1,0x8f81,0x4f40,0x8d01,0x4dc0,0x4c80,0x8c41,
	0x4400,0x84c1,0x8581,0x4540,0x8701,0x47c0,0x4680,0x8641,0x8201,0x42c0,0x4380,0x8341,0x4100,0x81c1,0x8081,0x4040,
};

int crc16(u16 *crc, u8 *buf, uint len)
{
	uint i;
	u16 sum = *crc;
	for(i = 0; i < len; i++) {
		sum = table_crc16[(buf[i] ^ sum) & 255] ^ (sum >> 8);
	}
	*crc = sum;
	return(0);
}

/* *ah* loads of global variables */

static u32 blocksize;
static u32 control; /* should be in a register */
static int shift; /* should be in a register */

static u8 *srcbuf, *dstbuf;
static uint src, dst;
static uint srclen, dstlen;

#define SHIFT() { if(shift > 0) { control |= (srcbuf[src] << 8 | srcbuf[src + 1]) << shift; src += 2; shift -= 16; } }
#define BITS(cnt) { bits = control >> (32 - (cnt)); shift += (cnt); control <<= (cnt); }

static u16 lit_codes[8192];
static u16 off_codes[1024];
static u8 lit_lens[512];
static u8 off_lens[32];
static u16 freq[17];
static u16 pref[17];

static int tree0(), tree1(), tree2(), tree3(), tree4();
static int huffman_prefixes();
static void general_tree(u16 *codes, uint base_size, u8 *lens, uint symbol);
static void special_tree(u16 *codes, uint base_size, u8 *lens, uint symbol);

//FILE * stderr;

int unlh5(u8 *srcpt, uint srcln, u8 *dstpt, uint dstln)
{
	srcbuf = srcpt;
	srclen = srcln + 6; /* little bit of unavoidable overrun */
	dstbuf = dstpt;
	dstlen = dstln;
	src = 0;
	dst = 0;
	control = 0;
	shift = 16;
	while(src < srclen && dst < dstlen) {
		if(tree0() || tree1() || tree2() || tree3() || tree4()) return(1); /* nasty */
	}
	return(dst != dstlen);
}

static int tree0()
{
	uint bits;
	SHIFT();
	BITS(16); blocksize = bits; SHIFT();
	if(src >= srclen) return(1);
	return(blocksize == 0);
}
static int tree1()
{
	uint bits, i, sym, len;
	BITS(5); sym = bits; SHIFT(); /* number of symbols defined in this tree */
	if(src >= srclen) return(1);
	if(sym == 0) {
		BITS(5); sym = bits; SHIFT();
		if(sym >= 20) return(1);
		special_tree(off_codes, 6, off_lens, sym);
		return(0);
	}
	if(sym >= 20) return(1);
	memset(off_lens, 0, sizeof(off_lens));
	memset(freq, 0, sizeof(freq));
	for(i = 0; i < sym; i++) {
		if(src >= srclen) return(1);
		BITS(3); len = bits;
		while(len >= 7) {
			BITS(1); if(bits == 0) break;
			len++;
			if(len >= 15) return(1);
		}
		SHIFT();
		off_lens[i] = len; freq[len]++;
		if(i == 2) { BITS(2); i += bits; SHIFT(); }
	}
	if(i != sym) return(1);
	if(huffman_prefixes()) return(1);
	memset(off_codes, 0, sizeof(off_codes));
	general_tree(off_codes, 6, off_lens, 20);
	return(0);
}
static int tree2()
{
	uint bits, i, sym, tmp;
	BITS(9); sym = bits; SHIFT(); /* number of symbols defined in this tree */
	if(src >= srclen) return(1);
	if(sym == 0) {
		BITS(9); sym = bits; SHIFT();
		if(sym >= 511) return(1);
		special_tree(lit_codes, 12, lit_lens, sym);
		return(0);
	}
	if(sym >= 511) return(1);
	memset(lit_lens, 0, sizeof(lit_lens));
	memset(freq, 0, sizeof(freq));
	for(i = 0; i < sym; i++) {
		if(src >= srclen) return(1);
		tmp = off_codes[control >> 26];
		if(tmp >= 64) {
			tmp = off_codes[tmp + (control >> 22 & 15)];
			if(tmp >= 64) {
				tmp = off_codes[tmp + (control >> 18 & 15)];
			}
		}
		shift += off_lens[tmp]; control <<= off_lens[tmp];
		SHIFT();
		switch(tmp) {
		case 0: break; /* zero */
		case 1: BITS(4); i += bits + 2; SHIFT(); break; /* zero 3-18 */
		case 2: BITS(9); i += bits + 19; SHIFT(); break; /* zero 20-531 */
		default: tmp -= 2; lit_lens[i] = tmp; freq[tmp]++; break;
		}
	}
	if(i != sym) return(1);
	if(huffman_prefixes()) return(1);
	memset(lit_codes, 0, sizeof(lit_codes));
	general_tree(lit_codes, 12, lit_lens, 510);
	return(0);
}
static int tree3()
{
	uint bits, i, sym, len;
	BITS(4); sym = bits; SHIFT(); /* number of symbols defined in this tree */
	if(src >= srclen) return(1);
	if(sym == 0) {
		BITS(4); sym = bits; SHIFT();
		if(sym >= 15) return(1);
		special_tree(off_codes, 8, off_lens, sym);
		return(0);
	}
	if(sym >= 15) return(1);
	memset(off_lens, 0, sizeof(off_lens));
	memset(freq, 0, sizeof(freq));
	for(i = 0; i < sym; i++) {
		if(src >= srclen) return(1);
		BITS(3); len = bits;
		while(len >= 7) {
			BITS(1); if(bits == 0) break;
			len++;
			if(len >= 17) return(1);
		}
		SHIFT();
		off_lens[i] = len; freq[len]++;
	}
	if(i != sym) return(1);
	if(huffman_prefixes()) return(1);
	memset(off_codes, 0, sizeof(off_codes));
	general_tree(off_codes, 8, off_lens, 15);
	return(0);
}
static int tree4()
{
	uint lit, len, off;
	while(blocksize--) {
		if(src >= srclen || dst >= dstlen) return(1);
		lit = lit_codes[control >> 20];
		if(lit >= 4096) {
			lit = lit_codes[lit + (control >> 16 & 15)];
		}
		shift += lit_lens[lit]; control <<= lit_lens[lit];
		SHIFT();
		if(lit < 256) { dstbuf[dst++] = lit; continue; }
		len = off_codes[control >> 24];
		if(len >= 256) {
			len = off_codes[len + (control >> 20 & 15)];
			if(len >= 256) {
				len = off_codes[len + (control >> 16 & 15)];
			}
		}
		shift += off_lens[len]; control <<= off_lens[len];
		SHIFT();
		if(len < 2) {
			off = len + 1;
		} else {
			len--;
			off = (1u << len | control >> (32 - len)) + 1;
			shift += len; control <<= len;
			SHIFT();
		}
		if(off > dst) return(1);
		off = dst - off;
		lit -= 253;
		while(lit--) dstbuf[dst++] = dstbuf[off++];
	}
	return(0);
}
static int huffman_prefixes()
{
	uint i;
	uint base = 0, left = 1;
	for(i = 1; i < 17; i++) {
		base *= 2; left *= 2;
		pref[i] = base;
		base += freq[i]; left -= freq[i];
	}
	return(left != 0);
}
static void special_tree(u16 *codes, uint base_size, u8 *lens, uint symbol)
{
	uint i;
	for(i = 0; i < (1u << base_size); i++) codes[i] = symbol; lens[symbol] = 0;
}
static void general_tree(u16 *codes, uint base_size, u8 *lens, uint symbol)
{
	uint i;
	uint len, code, base, bits;
	uint alloc;
	alloc = 1u << base_size;
	for(i = 0; i < symbol; i++) {
		len = lens[i];
		if(len == 0) continue;
		code = pref[len]++;
		base = 0;
		bits = base_size;
		while(len > bits) {
			len -= bits;
			base += code >> len;
			code &= (1u << len) - 1;
			bits = 4;
			if(codes[base] == 0) {
				codes[base] = alloc; alloc += 1u << bits;
			}
			base = codes[base];
		}
		if(len < bits) {
			bits -= len;
			base += code << bits;
			for(len = 1u << bits; len--; base++) codes[base] = i;
		} else {
			codes[base + code] = i;
		}
	}
}

/* yuck but seems to work */
#ifdef __cplusplus
extern "C" { extern int getopt(int,char**,char*); extern int mkdir(char*,int); }
#endif


char *progname;
int verbose;
char *name;
FILE *file;
char *out_name;
FILE *out_file;

uint block_size = 16384; /* default block size for output hdf */

#define MAX_BLOCKS 512 /* maximum number of blocks allowed in file */

static u8 buf[65536];
static u32 seek;

int read_block()
{
	u16 len = 0, crc = 0, sum = 0;
	if(fseek(file, seek, SEEK_SET) == -1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "seek to block failed", name, seek, strerror(errno));
		return(1);
	}
	if(fread(buf, 4, 1, file) != 1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "read block header failed", name, seek, strerror(errno));
		return(1);
	}
	len = HDR_LENGTH(buf) + 2;
	if(fread(buf + 4, len - 4, 1, file) != 1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "read block data failed", name, seek, strerror(errno));
		return(1);
	}
	crc = HDR_CRC(buf);
	crc16(&sum, buf + 4, len - 4);
	if(crc != sum) {
		fprintf(stderr,"%s: %s (%s:0x%lx): 0x%04x!=0x%04x\n", progname, "block checksum failed", name, seek, crc, sum);
		return(1);
	}
	seek += len;
	return(0);
}

struct info {
	u16 model;
	u16 blocks;
	u32 id1;
	u32 id2;
} info;

struct block {
	u32 seek;
	u16 type;
	u16 length;
	u32 address;
	u16 packlen;
	uint packed;
} block[MAX_BLOCKS];

static int read_info()
{
	if(read_block()) return(1);
	if(BLOCK_IS_INFO(buf)) {
		info.model = INFO_MODEL(buf);
		info.blocks = INFO_BLOCKS(buf);
		info.id1 = INFO_ID1(buf);
		info.id2 = INFO_ID2(buf);
	} else if(BLOCK_IS_INFO2(buf)) {
		info.model = INFO2_MODEL(buf);
		info.blocks = INFO2_BLOCKS(buf);
		info.id1 = info.id2 = INFO2_ID(buf);
	} else {
		fprintf(stderr, "%s: %s (%s)\n", progname, "unknown or unsupported hdf file", name);
		return(1);
	}
	if(verbose) fprintf(stderr, "info: model %x id1 %lx id2 %lx\n", info.model, info.id1, info.id2);
	if(info.blocks > MAX_BLOCKS) {
		fprintf(stderr, "%s: %s (%s): %d>%d\n", progname, "too many blocks to handle", name, info.blocks, MAX_BLOCKS);
		return(1);
	}
	return(0);
}

static int read_data()
{
	uint i;
	for(i = 0; i < info.blocks; i++) {
		if(verbose) fprintf(stderr, "block %d: ", i);
		block[i].seek = seek;
		if(read_block()) return(1);
		block[i].type = DATA_TYPE(buf);
		block[i].length = DATA_LENGTH(buf);
		block[i].address = DATA_ADDRESS(buf);
		block[i].packlen = DATA_PACKLEN(buf);
		block[i].packed = DATA_IS_PACKED(buf);
		if(verbose) fprintf(stderr, "type %x address 0x%lx length 0x%x\n", block[i].type, block[i].address, block[i].length);
	}
	return(0);
}

static u16 type;
static u32 start;

static void block_summary()
{
	uint i;
	printf("flash summary:\n");
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		printf("type %x: address 0x%lx", type, start);
		do {
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		printf(" - 0x%lx\n", start - 1);
	}
}

static u8 unpacked[65536];
char out_path[4096];
#define snprintf _snprintf
static int dump_files(CString out_dir,int savehdf)
{
	uint i;
	uint num;
	num = 0;
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		if(savehdf){
			snprintf(out_path, sizeof(out_path), "%s/hdf-%x-%06x.hdf", out_dir, type, start);		
		}else{
			snprintf(out_path, sizeof(out_path), "%s/hdfbin-%x-%06x.raw", out_dir, type, start);
		}
		out_name = out_path;
		out_file = fopen(out_name, "wb");
		if(out_file == 0) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for writing failed", out_name, strerror(errno));
			return(1);
		}
		do {
			seek = block[i].seek;
			if(read_block()) return(1);
			if(block[i].packed) {
				if(unlh5(buf + DATA_size, block[i].packlen, unpacked, block[i].length)) {
					fprintf(stderr, "%s: %s (%s:0x%lx)\n", progname, "failed to unpack block", name, block[i].seek);
					return(1);
				}
			} else {
				memcpy(unpacked, buf + DATA_size, block[i].length);
			}
			if(fwrite(unpacked, block[i].length, 1, out_file) != 1) {
				fprintf(stderr, "%s: %s (%s): %s\n", progname, "write unpacked data failed", out_name, strerror(errno));
				return(1);
			}
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		fclose(out_file); out_file = 0;
		num++;
	}
	return(0);
}

static u8 out_block[65536];
static u32 block_start;
static uint block_pos;

static int write_info(int savehdf)
{
	uint i;
	uint info_blocks = 0;
	u32 info_length = 0;
	u32 length;
	u16 sum;
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		length = 0; /* length of data in this group */
		do {
			length += block[i].length;
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		info_blocks += (length + block_size - 1) / block_size;
		info_length += length;
	}
	info_length += INFO_size + info_blocks * DATA_size;
	buf[0] = 0; buf[1] = 18;
	buf[4] = info.model >> 8; buf[5] = info.model;
	buf[6] = info_blocks >> 8; buf[7] = info_blocks;
	buf[8] = info.id1 >> 24; buf[9] = info.id1 >> 16; buf[10] = info.id1 >> 8; buf[11] = info.id1;
	buf[12] = info.id2 >> 24; buf[13] = info.id2 >> 16; buf[14] = info.id2 >> 8; buf[15] = info.id2;
	buf[16] = info_length >> 24; buf[17] = info_length >> 16; buf[18] = info_length >> 8; buf[19] = info_length;
	sum = 0; crc16(&sum, buf + 4, 16); buf[2] = sum >> 8; buf[3] = sum;
	if(savehdf){
		if(fwrite(buf, 20, 1, out_file) != 1) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "write header failed", out_name, strerror(errno));
			return(1);
		}
	}
	if(verbose) fprintf(stderr, "info: model %x id1 %lx id2 %lx\n", info.model, info.id1, info.id2);
	return(0);
}

static int write_block(int savehdf)
{
	u16 sum;
	buf[0] = (block_pos + 10) >> 8; buf[1] = (block_pos + 10);
	buf[4] = type >> 8; buf[5] = type;
	buf[6] = block_pos >> 8; buf[7] = block_pos;
	buf[8] = block_start >> 24; buf[9] = block_start >> 16; buf[10] = block_start >> 8; buf[11] = block_start;
	sum = 0; crc16(&sum, buf + 4, 8); crc16(&sum, out_block, block_pos); buf[2] = sum >> 8; buf[3] = sum;
	if(savehdf){
		if(fwrite(buf, 12, 1, out_file) != 1) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "write block header failed", out_name, strerror(errno));
			return(1);
		}
	}
	if(fwrite(out_block, block_pos, 1, out_file) != 1) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "write block data failed", out_name, strerror(errno));
		return(1);
	}
	if(verbose) fprintf(stderr, "type %x address 0x%lx length 0x%x\n", type, block_start, block_pos);
	block_start += block_pos;
	block_pos = 0;
	return(0);
}

#undef min
#define min(a,b) ((a) < (b) ? (a) : (b))

static int write_data(int savehdf)
{
	uint i;
	uint j = 0;
	uint pos;
	uint size;

	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		block_start = start;
		block_pos = 0;
		do {
			seek = block[i].seek;
			if(read_block()) return(1);
			if(block[i].packed) {
				if(unlh5(buf + DATA_size, block[i].packlen, unpacked, block[i].length)) {
					fprintf(stderr, "%s: %s (%s:0x%lx)\n", progname, "failed to unpack block", name, block[i].seek);
					return(1);
				}
			} else {
				memcpy(unpacked, buf + DATA_size, block[i].length);
			}
			pos = 0;
			while(pos < block[i].length) {
				size = min(block_size - block_pos, block[i].length - pos);
				memcpy(out_block + block_pos, unpacked + pos, size);
				pos += size;
				block_pos += size;
				if(block_pos == block_size) {
					if(verbose) fprintf(stderr, "block %d: ", j);
					if(write_block(savehdf)) return(1);
					j++;
				}
			}
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		if(block_pos > 0) {
			if(verbose) fprintf(stderr, "block %d: ", j);
			if(write_block(savehdf)) return(1);
			j++;
		}
	}
	return(0);
}

static int sort_blocks(const void *aa, const void *bb)
{
	const struct block *a = (struct block *)aa, *b = (struct block *)bb;
	if(a->type != b->type) return((a->type > b->type) - (a->type < b->type));
	return((a->address > b->address) - (a->address < b->address));
}

int uhdfmain(CString infile, CString outfile, int blocksize,CString dirfile,int blocksort,int splitblocks,int savehdf)
{
/*
		printf("usage: %s [options] hdf-file\n", progname);
		printf("\t-o file  output unpacked hdf to this file\n");
		printf("\t-b size  make data blocks this size (%d)\n", block_size);
		printf("\t-d dir   output data files to this directory\n");
		printf("\t-h       sort data blocks\n");

*/
	if(blocksize) {
		if(blocksize < 4096 || blocksize > 65520) {
			fprintf(stderr, "%s: %s (4096<%d<65520)\n", progname, "block size out of range", blocksize);
			goto fail;
		}
	}
	
	file = fopen(infile, "rb");
	if(file == 0) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for reading failed", name, strerror(errno));
		goto fail;
	}

	seek = 0;
	if(read_info()) goto fail;
	if(read_data()) goto fail;
	if(blocksort) qsort(block, info.blocks, sizeof(*block), sort_blocks);
	block_summary();
	if(splitblocks) {
		if(dump_files(dirfile,savehdf)) goto fail;
	} else {
		out_file = fopen(dirfile+"\\"+outfile, "wb");
		if(out_file == 0) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for writing failed", out_name, strerror(errno));
			goto fail;
		}
		if(write_info(savehdf)) goto fail;
		if(write_data(savehdf)) goto fail;
	}
fail:
	if(file) fclose(file);
	if(out_file) fclose(out_file);
	return(0);
}





