/* unpack a hdf file
 *
 * 2003-02-12T06:57:49Z
 *   works
 *
 * 2003-02-06T23:11:25Z
 */

#include "hdf.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

/* yuck but seems to work */
#ifdef __cplusplus
extern "C" { extern int getopt(int,char**,char*); extern int mkdir(char*,int); }
#endif

extern int unlh5(u8 *src, uint srclen, u8 *dst, uint dstlen);
extern int crc16(u16 *crc, u8 *buf, uint len);

char *progname;
int verbose;
char *name;
FILE *file;
char *out_name;
FILE *out_file;

uint block_size = 16384; /* default block size for output hdf */

#define MAX_BLOCKS 512 /* maximum number of blocks allowed in file */

static u8 buf[65536];
static u32 seek;

int read_block()
{
	u16 len = 0, crc = 0, sum = 0;
	if(fseek(file, seek, SEEK_SET) == -1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "seek to block failed", name, seek, strerror(errno));
		return(1);
	}
	if(fread(buf, 4, 1, file) != 1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "read block header failed", name, seek, strerror(errno));
		return(1);
	}
	len = HDR_LENGTH(buf) + 2;
	if(fread(buf + 4, len - 4, 1, file) != 1) {
		fprintf(stderr, "%s: %s (%s:0x%lx): %s\n", progname, "read block data failed", name, seek, strerror(errno));
		return(1);
	}
	crc = HDR_CRC(buf);
	crc16(&sum, buf + 4, len - 4);
	if(crc != sum) {
		fprintf(stderr,"%s: %s (%s:0x%lx): 0x%04x!=0x%04x\n", progname, "block checksum failed", name, seek, crc, sum);
		return(1);
	}
	seek += len;
	return(0);
}

struct info {
	u16 model;
	u16 blocks;
	u32 id1;
	u32 id2;
} info;

struct block {
	u32 seek;
	u16 type;
	u16 length;
	u32 address;
	u16 packlen;
	uint packed;
} block[MAX_BLOCKS];

static int read_info()
{
	if(read_block()) return(1);
	if(BLOCK_IS_INFO(buf)) {
		info.model = INFO_MODEL(buf);
		info.blocks = INFO_BLOCKS(buf);
		info.id1 = INFO_ID1(buf);
		info.id2 = INFO_ID2(buf);
	} else if(BLOCK_IS_INFO2(buf)) {
		info.model = INFO2_MODEL(buf);
		info.blocks = INFO2_BLOCKS(buf);
		info.id1 = info.id2 = INFO2_ID(buf);
	} else {
		fprintf(stderr, "%s: %s (%s)\n", progname, "unknown or unsupported hdf file", name);
		return(1);
	}
	if(verbose) fprintf(stderr, "info: model %x id1 %lx id2 %lx\n", info.model, info.id1, info.id2);
	if(info.blocks > MAX_BLOCKS) {
		fprintf(stderr, "%s: %s (%s): %d>%d\n", progname, "too many blocks to handle", name, info.blocks, MAX_BLOCKS);
		return(1);
	}
	return(0);
}

static int read_data()
{
	uint i;
	for(i = 0; i < info.blocks; i++) {
		if(verbose) fprintf(stderr, "block %d: ", i);
		block[i].seek = seek;
		if(read_block()) return(1);
		block[i].type = DATA_TYPE(buf);
		block[i].length = DATA_LENGTH(buf);
		block[i].address = DATA_ADDRESS(buf);
		block[i].packlen = DATA_PACKLEN(buf);
		block[i].packed = DATA_IS_PACKED(buf);
		if(verbose) fprintf(stderr, "type %x address 0x%lx length 0x%x\n", block[i].type, block[i].address, block[i].length);
	}
	return(0);
}

static u16 type;
static u32 start;

static void block_summary()
{
	uint i;
	printf("flash summary:\n");
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		printf("type %x: address 0x%lx", type, start);
		do {
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		printf(" - 0x%lx\n", start - 1);
	}
}

static u8 unpacked[65536];
static char out_path[4096];

static int dump_files(char *out_dir)
{
	uint i;
	uint num;
	num = 0;
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		snprintf(out_path, sizeof(out_path), "%s/file-%u-type-%x-start-%lx", out_dir, num, type, start);
		out_name = out_path;
		out_file = fopen(out_name, "wb");
		if(out_file == 0) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for writing failed", out_name, strerror(errno));
			return(1);
		}
		do {
			seek = block[i].seek;
			if(read_block()) return(1);
			if(block[i].packed) {
				if(unlh5(buf + DATA_size, block[i].packlen, unpacked, block[i].length)) {
					fprintf(stderr, "%s: %s (%s:0x%lx)\n", progname, "failed to unpack block", name, block[i].seek);
					return(1);
				}
			} else {
				memcpy(unpacked, buf + DATA_size, block[i].length);
			}
			if(fwrite(unpacked, block[i].length, 1, out_file) != 1) {
				fprintf(stderr, "%s: %s (%s): %s\n", progname, "write unpacked data failed", out_name, strerror(errno));
				return(1);
			}
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		fclose(out_file); out_file = 0;
		num++;
	}
	return(0);
}

static u8 out_block[65536];
static u32 block_start;
static uint block_pos;

static int write_info()
{
	uint i;
	uint info_blocks = 0;
	u32 info_length = 0;
	u32 length;
	u16 sum;
	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		length = 0; /* length of data in this group */
		do {
			length += block[i].length;
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		info_blocks += (length + block_size - 1) / block_size;
		info_length += length;
	}
	info_length += INFO_size + info_blocks * DATA_size;
	buf[0] = 0; buf[1] = 18;
	buf[4] = info.model >> 8; buf[5] = info.model;
	buf[6] = info_blocks >> 8; buf[7] = info_blocks;
	buf[8] = info.id1 >> 24; buf[9] = info.id1 >> 16; buf[10] = info.id1 >> 8; buf[11] = info.id1;
	buf[12] = info.id2 >> 24; buf[13] = info.id2 >> 16; buf[14] = info.id2 >> 8; buf[15] = info.id2;
	buf[16] = info_length >> 24; buf[17] = info_length >> 16; buf[18] = info_length >> 8; buf[19] = info_length;
	sum = 0; crc16(&sum, buf + 4, 16); buf[2] = sum >> 8; buf[3] = sum;
	if(fwrite(buf, 20, 1, out_file) != 1) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "write header failed", out_name, strerror(errno));
		return(1);
	}
	if(verbose) fprintf(stderr, "info: model %x id1 %lx id2 %lx\n", info.model, info.id1, info.id2);
	return(0);
}

static int write_block()
{
	u16 sum;
	buf[0] = (block_pos + 10) >> 8; buf[1] = (block_pos + 10);
	buf[4] = type >> 8; buf[5] = type;
	buf[6] = block_pos >> 8; buf[7] = block_pos;
	buf[8] = block_start >> 24; buf[9] = block_start >> 16; buf[10] = block_start >> 8; buf[11] = block_start;
	sum = 0; crc16(&sum, buf + 4, 8); crc16(&sum, out_block, block_pos); buf[2] = sum >> 8; buf[3] = sum;
	if(fwrite(buf, 12, 1, out_file) != 1) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "write block header failed", out_name, strerror(errno));
		return(1);
	}
	if(fwrite(out_block, block_pos, 1, out_file) != 1) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "write block data failed", out_name, strerror(errno));
		return(1);
	}
	if(verbose) fprintf(stderr, "type %x address 0x%lx length 0x%x\n", type, block_start, block_pos);
	block_start += block_pos;
	block_pos = 0;
	return(0);
}

#undef min
#define min(a,b) ((a) < (b) ? (a) : (b))

static int write_data()
{
	uint i;
	uint j = 0;
	uint pos;
	uint size;

	for(i = 0; i < info.blocks;) {
		type = block[i].type;
		start = block[i].address;
		block_start = start;
		block_pos = 0;
		do {
			seek = block[i].seek;
			if(read_block()) return(1);
			if(block[i].packed) {
				if(unlh5(buf + DATA_size, block[i].packlen, unpacked, block[i].length)) {
					fprintf(stderr, "%s: %s (%s:0x%lx)\n", progname, "failed to unpack block", name, block[i].seek);
					return(1);
				}
			} else {
				memcpy(unpacked, buf + DATA_size, block[i].length);
			}
			pos = 0;
			while(pos < block[i].length) {
				size = min(block_size - block_pos, block[i].length - pos);
				memcpy(out_block + block_pos, unpacked + pos, size);
				pos += size;
				block_pos += size;
				if(block_pos == block_size) {
					if(verbose) fprintf(stderr, "block %d: ", j);
					if(write_block()) return(1);
					j++;
				}
			}
			start += block[i].length;
		} while(++i < info.blocks && block[i].address == start && block[i].type == type);
		if(block_pos > 0) {
			if(verbose) fprintf(stderr, "block %d: ", j);
			if(write_block()) return(1);
			j++;
		}
	}
	return(0);
}

static int sort_blocks(const void *aa, const void *bb)
{
	const struct block *a = (struct block *)aa, *b = (struct block *)bb;
	if(a->type != b->type) return((a->type > b->type) - (a->type < b->type));
	return((a->address > b->address) - (a->address < b->address));
}

int main(int argc, char **argv)
{
	char *arg_file = 0;
	char *arg_size = 0;
	char *arg_dir = 0;
	int arg_sort = 0;
	progname = strrchr(argv[0], '/'); progname = progname ? progname + 1 : argv[0];
parse:
	switch(getopt(argc, argv, "o:b:d:hvV")) {
	extern int optind;
	extern char *optarg;
	case 'o': arg_file = optarg; goto parse;
	case 'b': arg_size = optarg; goto parse;
	case 'd': arg_dir = optarg; goto parse;
	case 'h': arg_sort = 1; goto parse;
	case 'v': verbose++; goto parse;
	case 'V':
		printf("unpackhdf 1.0\n");
		printf("unpack a humax data file\n");
		printf("hrac@address.com\n");
		return(0);
	default:
		printf("usage: %s [options] hdf-file\n", progname);
		printf("\t-o file  output unpacked hdf to this file\n");
		printf("\t-b size  make data blocks this size (%d)\n", block_size);
		printf("\t-d dir   output data files to this directory\n");
		printf("\t-h       sort data blocks\n");
		printf("\t-v       verbose\n");
		printf("\t-V       print version\n");
		return(1);
	case -1:
		argc -= optind;
		argv += optind;
	}
	if(argc != 1) {
		fprintf(stderr, "%s: %s\n", progname, "wrong number of arguments");
		goto fail;
	}
	if(arg_size) {
		char *endptr;
		block_size = strtoul(arg_size, &endptr, 0);
		if(*arg_size == 0 || *endptr != 0) {
			fprintf(stderr, "%s: %s (%s)\n", progname, "bad number", arg_size);
			goto fail;
		}
		if(block_size < 4096 || block_size > 65520) {
			fprintf(stderr, "%s: %s (4096<%d<65520)\n", progname, "block size out of range", block_size);
			goto fail;
		}
	}
	name = argv[0];
	file = fopen(name, "rb");
	if(file == 0) {
		fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for reading failed", name, strerror(errno));
		goto fail;
	}

	seek = 0;
	if(read_info()) goto fail;
	if(read_data()) goto fail;
	if(arg_sort) qsort(block, info.blocks, sizeof(*block), sort_blocks);
	block_summary();
	if(arg_dir) {
		mkdir(arg_dir, 511);
		if(dump_files(arg_dir)) goto fail;
	}
	if(arg_file) {
		out_name = arg_file;
		out_file = fopen(out_name, "wb");
		if(out_file == 0) {
			fprintf(stderr, "%s: %s (%s): %s\n", progname, "open for writing failed", out_name, strerror(errno));
			goto fail;
		}
		if(write_info()) goto fail;
		if(write_data()) goto fail;
	}
fail:
	if(file) fclose(file);
	if(out_file) fclose(out_file);
	return(0);
}
